#ifndef AHLGREN_WAM
#define AHLGREN_WAM

#include <stack> // PDL
#include <map>
#include <array>
#include <utility>
#include "term.h"
#include "tokenizer.h"
#include "parameters.h"
#include "mode.h"
#include "global.h"
#include "functor.h"

// For prompt
#include <string>
#include <iostream>
#include <fstream>

#include <iterator>

namespace lp {

	// Fixed Memory Sizes
	const int HEAP_SIZE = 4*1024*1024;
	const int STACK_SIZE = 4*1024*1024;
	const int VAR_SIZE = 1024*1024;
	const int CODE_AREA_SIZE = 4*1024*1024;
	const int QUERY_AREA_SIZE = 1024*1024;
	const int TRAIL_SIZE = 2*1024*1024;

	// Global Stack
	extern std::array<term,HEAP_SIZE> heap;
	extern int heap_i; // H
	extern int heap_back; // HB
	extern int env_i; // E
	extern int env_back; // B
	extern int env_back0; // B0
	extern int stack_bottom;
	extern int num_of_args;
	// Read/Write mode
	enum wam_mode_t { READ, WRITE };
	extern wam_mode_t wam_mode;
	extern term* s_ptr; // pointer to hs for reading arguments
	// Local Stack
	extern std::array<term,STACK_SIZE> stack;
	// Variable registers
	extern std::array<term*,VAR_SIZE> var_regs;
	// Trail
	extern std::array<term*,TRAIL_SIZE> trail;
	extern int tindex; // TR
	// Runtime info
	extern int qlevel; // query level
	extern int qlevel2;
	extern boost::filesystem::path CURRENT_PATH;

	// Substitutions in answers (used by WAM)
	//typedef std::map<lp::id_type,std::unique_ptr<Functor>> subst;
	typedef std::deque<Subst> answer_type;

	// Code Area
	struct instruction {
		enum opcode_t {
			STOP, NOOP, PROCEED, CALL, EXEC, DYN_CALL, DYN_EXEC, DYN_STACK_CALL, DYN_STACK_EXEC,
			GET_VAR, GET_VARY, GET_VAL, GET_VALY, GET_STR, GET_LIS, GET_CON, GET_INT, GET_FLT,  
			UNI_VAR, UNI_VAL, UNI_VARY, UNI_VALY, UNI_LOCAL_VAL, UNI_LOCAL_VALY, UNI_CON, UNI_VOID, UNI_INT, UNI_FLT, 
			PUT_VAR, PUT_VARY, PUT_VAL, PUT_VALY, PUT_UNSAFE_VALY, PUT_STR, PUT_LIS, PUT_CON, PUT_INT, PUT_FLT,
			SET_VAR, SET_VARY, SET_VAL, SET_VALY, SET_LOCAL_VAL, SET_LOCAL_VALY, SET_CON, SET_VOID, SET_INT, SET_FLT,
			REG_HVAR, REG_SVAR, REG_ARG, GLOBALIZE_SVARS, // register query variables / ptr to arguments
			ALLOC, DEALLOC, TRY_ME_ELSE, RETRY_ME_ELSE, TRUST_ME, PROMPT, CONTINUE,
			NECK_CUT, GET_LVL, CUT,
			UNIFY, FAIL, VAR, NONVAR, ATOMIC, ATOM, INTEGER, FLOAT, NUMBER, NONUNIFIABLE,
			EQUAL, UNEQUAL, UNIV, FUNCTOR, ARG, NAME,
			IS, NUMEQUAL, NUMUNEQUAL, NUMLESS, NUMLESSEQ, NUMGREATER, NUMGREATEREQ,
			BEFORE, BEFORE_EQ, AFTER, AFTER_EQ, COMPARE, VARIANT, NOT_VARIANT,
			LISTING, LISTING_WAM, LISTING1, LISTING_WAM1, WRITE, NL, CONSULT, CONSULT_LIST,
			HALT, ASSERT, ASSERTA, RETRACT, RETRACTALL, OP,
			NB_LINKARG, DUPE_TERM,
			MODEH, MODEB, MODEB3, SET_PARAM, GET_PARAM, DETERMINATION, EXCLUDE_PRED, GENERALIZE0, GENERALIZE1,
			IMPLIES, PREVENTS, PREVENT,
		};
		instruction() : opcode(STOP), a1(0), a2(0), a3(0) {}
		instruction(opcode_t opc) : opcode(opc) {}
		instruction(opcode_t opc, int x) : opcode(opc), a1(x) {}
		instruction(opcode_t opc, int x, int y) : opcode(opc), a1(x), a2(y) {}
		instruction(opcode_t opc, int x, int y, int z) : opcode(opc), a1(x), a2(y), a3(z) {}
		// For instructions containing numeric values (long long or double)
		instruction(opcode_t opc, long long x, int y = 0) : opcode(opc), i1(x), a2(y) {}
		instruction(opcode_t opc, double x, int y = 0) : opcode(opc), d1(x), a2(y) {}
		opcode_t opcode;
		union { int a1; long long i1; double d1; };
		int a2,a3;
	};
	std::ostream& operator<<(std::ostream&, const instruction&);


	// Typedefs
	typedef std::pair<lp::id_type,int> sign_t;
	typedef std::vector<instruction> clause;
	typedef std::vector<clause> concept;
	typedef std::map<sign_t,concept> code;
	// Pointer to knowledge base
	struct code_iterator : public std::iterator<std::bidirectional_iterator_tag,instruction> {
		//typedef std::bidirectional_iterator_tag iterator_category;
		//typedef instruction value_type;
		//typedef instruction* pointer;
		//typedef instruction& reference;
		code_iterator(concept* cp = nullptr, int i = 0, int l = 0) : con_ptr(cp), cl_i(i), line(l) {}
		// Increment/Decrement
		code_iterator& operator++() { ++line; return *this; }
		code_iterator operator++(int) { auto c = *this; ++(*this); return c; }
		code_iterator& operator--() { --line; return *this; }
		code_iterator operator--(int) { auto c = *this; --(*this); return c; }
		// Insert instruction
		code_iterator& operator<<(instruction& ic) {
			//std::cerr << "inserting instruction: " << ic << "\n";
			assert(con_ptr);
			clause& cref = (*con_ptr)[cl_i];
			cref.insert(cref.begin()+line,ic);
			++line;
			return *this;
		}
		code_iterator& operator<<(instruction&& ic) { 
			//std::cerr << "inserting instruction: " << ic << "\n";
			assert(con_ptr);
			clause& cref = (*con_ptr)[cl_i];
			cref.insert(cref.begin()+line,std::move(ic));
			++line;
			return *this;
		}
		// Jump to definition
		void reset(concept* cp = nullptr, int i = 0, int l = 0) { con_ptr = cp; cl_i = i; line = l; }
		void reset(clause* clp, int l = 0) { con_ptr = nullptr; cl_ptr = clp; line = l; }
		instruction& operator*() {
			assert( !con_ptr || !con_ptr->empty() );
			assert( !con_ptr || cl_i < int(con_ptr->size()) );
			assert( !con_ptr || line < int((*con_ptr)[cl_i].size()) );
			return con_ptr ? ((*con_ptr)[cl_i])[line] : (*cl_ptr)[line];
		}
		instruction* operator->() { return &(this->operator*()); }
		// Data members
		concept* con_ptr; // which concept? (nullptr <=> external query)
		union { // which clause?
			int cl_i; 
			clause* cl_ptr; // for external query
		};
	 	int line; // code line in clause
	};
	extern code_iterator cap;
	extern code_iterator cap2;

	// Variable mapping
	extern std::map<id_type,term*> var_map;
	extern std::vector<term*> arg_map;

	// Fail exception
	class wam_fail {};

	// WAM State
	struct wam_state {
		code_iterator cap,cap2;
		decltype(var_map) var_map;
		decltype(arg_map) arg_map;
		int heap_i,heap_back,env_i,env_back,env_back0,
			stack_bottom,num_of_args,tindex,qlevel,qlevel2;
		boost::filesystem::path cwd;
	};


	// Query constraints
	struct qconstraints {
		qconstraints() : depth(numeric_limits<int>::max()), recall(numeric_limits<int>::max()),
			resolutions(numeric_limits<int>::max()), instructions(numeric_limits<int>::max()), update_labels(0), interactive(false) {}
		qconstraints(const parameters& p) : depth(p.force_int(parameters::qdepth)), recall(numeric_limits<int>::max()),
			resolutions(p.force_int(parameters::qres)), instructions(p.force_int(parameters::max_instructions)),
			update_labels(p.is_set(parameters::update_labels)), interactive(false) {}
		int depth, recall, resolutions, instructions, update_labels;
		bool interactive;
	};

	//================= Function Declarations =================//
	// Helper Functions
	inline bool is_stack(const term* v) { return &lp::stack[0] <= v && v <= &lp::stack[lp::stack.size()-1]; }
	inline bool is_heap(const term* v) { return &lp::heap[0] <= v && v <= &lp::heap[lp::heap.size()-1]; }
	// Unify using trail
	bool unify(term* x, term* y);
	// Unify without using trail
	bool unify_external(term* x, term* y, std::map<term*,term*>& bindings);
	// Test unifiability without using trail or actually binding anything
	inline bool unifiable(const term* x, const term* y) { std::map<term*,term*> b; return unify_external(const_cast<term*>(x), const_cast<term*>(y),b); }
	void update_trail(term* p);
	void unwind_trail(int beg, int end);
	void tidy_trail();
	void bind(term* x, term* y);

	std::vector<const Functor*> make_equations(const Functor& t, std::map<const Functor*,int>& vlink);

	//void wam_print(std::ostream&);

	//=================== Convert Tree to WAM =========================//
	using lp::id_type;

	// Extract WAM Variable Ordering from Clause
	struct vinfo {
		vinfo() : var(0), first(-1), last(-1), nr(0), count(0), head_upos(-1), body_upos(-1), temporary(false), nested(false) {}
		vinfo(id_type v, int i, bool nest, int hp = 0) 
			: var(v), first(i), last(i), nr(++counter), count(1), head_upos(hp), body_upos(-1), temporary(false), nested(nest) {}
		unsigned long long nr;
		id_type var;
		int first, last, count, index;
		int head_upos,body_upos; // unique arg position in head and first body? top = 0, invalid = -1
		bool temporary;
		bool nested; // is ANY occurrence nested?
		static unsigned long long counter;
	};
	// Sort vinfo for permanent variables
	struct cmp_permanent {
		bool operator()(const vinfo* v, const vinfo* w) const {
			if (v->last > w->last) return true;
			if (v->last < w->last) return false;
			return v->nr < w->nr; // pick whichever came first
		}
	};
	// Sort vinfo for temporary variables
	struct cmp_temporary {
		bool operator()(const vinfo* v, const vinfo* w) const {
			return v->nr < w->nr; // pick whichever came first
		}
	};
	// Define functions for creating variable indices
	typedef std::pair<std::set<vinfo,cmp_temporary>,std::set<vinfo,cmp_permanent>> var_tbl_type;
	std::map<id_type,vinfo> make_variable_table(const Functor& f, bool collect, int& permc, int& tempc, int& max_argc, int& cl_len, int& last_cut);

	// Flatten Clause
	//std::vector<const Functor*> make_equations(const Functor& t);

	// Print instruction
	std::ostream& operator<<(std::ostream& os, const instruction& i);

	// Table lookup for built-in predicates
	typedef map<std::pair<lp::id_type,int>,lp::instruction::opcode_t> builtin_lookup_type;
	extern builtin_lookup_type builtin;
	builtin_lookup_type define_builtins(); // helper to create builtins

	// Clear WAM state
	//void wam_clear();

	// Compile to target clause*
	void make_wam_instructions(const Functor& cl, const std::set<id_type>&, bool collect, clause*);
	inline void make_wam_instructions(const Functor& cl, bool c, clause* ci) { std::set<id_type> m; return make_wam_instructions(cl,m,c,ci); }
	inline clause compile(const Functor& f, bool c = false) { clause cl; make_wam_instructions(f,c,&cl); return cl; }

	// Decompile instruction into Functor
	Functor decompile(const sign_t&, const clause&);
	Functor decompile(const clause&); // for queries
	Functor decompile(const sign_t& head_sign, const instruction* i);
	Functor decompile(const instruction* i); // No head
	//inline Functor decompile(const sign_t& s, const instruction* i) { Subst v; return decompile(s,i,v); }
	//inline Functor decompile(const instruction* i) { Subst v; return decompile(i,v); }

	// Exception thrown if term is cyclic
	struct cyclic_term : public std::exception { 
		cyclic_term(const term* n = nullptr) : std::exception(), ptr(n) {}
		const term* ptr;
	};
	// Convert term to functor
	Functor* term2functor(const term* t, std::map<const term*,id_type>& vmap);
	inline Functor* term2functor(const term* t) { std::map<const term*,id_type> m; return term2functor(t,m); }

	// Is instruction builtin?
	inline bool is_builtin(const instruction& i) { 
		return std::find_if(builtin.begin(),builtin.end(),[&i](const builtin_lookup_type::value_type& p){
			return p.second == i.opcode; }) != builtin.end(); 
	}

	inline bool end_of_literal(const instruction& i)
	{
		switch (i.opcode) {
		case instruction::CALL: case instruction::EXEC:
		case instruction::DYN_CALL: case instruction::DYN_EXEC:
		case instruction::DYN_STACK_CALL: case instruction::DYN_STACK_EXEC:
			return true;
		case instruction::NOOP: return false;
		default: return is_builtin(i);
		}
	}

	// Count number of literals
	inline int body_literals(const clause& cl)
	{
		return std::count_if(cl.begin(),cl.end(),[](const instruction& i){return end_of_literal(i);});
	}

	// Ground?
	bool is_ground(const instruction* i);
	bool is_ground(const sign_t& s, const instruction* i);
	inline bool is_ground(const clause& cl) { return is_ground(&cl.front()); }
	inline bool is_ground(const sign_t& s, const clause& cl) { return is_ground(s,&cl.front()); }


	inline Functor decompile(const clause& cl) { return decompile(&cl.front()); }
	inline Functor decompile(const sign_t& s, const clause& cl) { return decompile(s,&cl.front()); }

}


#endif

